﻿using System;
using System.IO;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters;
using System.Runtime.Serialization.Formatters.Binary;
using System.Workflow.ComponentModel;
using System.Workflow.ComponentModel.Serialization;
using System.Workflow.Runtime.Hosting;

namespace Lessons.Workflows
{
    /* a custom persistence service class which saves workflow instances to disk files */
    public class CustomPersistenceService : WorkflowPersistenceService
    {
        private string folder;
        private bool unloadOnIdle = false;

        /* creates a full filename for the serialization data file */
        string CreateFileName(Guid ID)
        {
            return Path.Combine(folder, ID.ToString() + ".wwf");
        }


        /* serializes and saves the activity to a disk file.     
     
            There may be situations where multiple instances of the workflow runtime engine share the
            same datastore when persisting workflows. WWF provides a locking mechanism in order to prevent
            persistence services that run in two different processes from loading the same workflow instance 
            into memory at the same time. If a custom persistence service is to support that locking mechanism 
            then the LoadWorkflowInstanceState() method must throw a WorkflowOwnershipException if the workflow
            being loaded is locked by another process. It also must lock the datastore while loading the workflow.
            The UnlockWorkflowInstanceState() method unlocks a previously locked instance. Also the SaveWorkflowInstanceState() 
            method provides the unlock boolean parameter to indicate whether state information of a workflow instance 
            being serialized should remain in an un-locked state while in the datastore. 
         */
        void Save(Activity activity, bool unlock)
        {
            Guid ID = (Guid)activity.GetValue(Activity.ActivityContextGuidProperty);

            string FileName = this.CreateFileName(ID);

            if (File.Exists(FileName))
                File.Delete(FileName);

            using (FileStream FS = new FileStream(FileName, FileMode.CreateNew))
            {
                /*  
                    Serialization surrogates are classes that implement the ISerializationSurrogate interface 
                    and they know how to serialize objects of a some particular type. SurrogateSelector
                    classes are selector objects that assist formatters in selecting a proper serialization
                    surrogate for a certain type. The ActivitySurrogateSelector class is a surrogate selector
                    for Activity classes. The ActivitySurrogateSelector.Default property provides a default
                    internal implementation of a serialization surrogate
                 */
                IFormatter formatter = new BinaryFormatter();
                formatter.SurrogateSelector = ActivitySurrogateSelector.Default;
                activity.Save(FS, formatter);
            }


            if (!unlock)
                File.SetAttributes(FileName, FileAttributes.ReadOnly);
        }


        /* locates a disk file, based on the ID, and loads the data to the activity */
        Activity Load(Guid ID, Activity activity)
        {
            string FileName = this.CreateFileName(ID);

            using (FileStream FS = new FileStream(FileName, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                /* see the comment for surrogates in the Save() method */
                FS.Position = 0;
                IFormatter formatter = new BinaryFormatter();
                formatter.SurrogateSelector = ActivitySurrogateSelector.Default;
                return Activity.Load(FS, activity, formatter);
            }
        }



        public CustomPersistenceService(string Folder)
        {
            this.folder = Folder;
        }

        public CustomPersistenceService(string Folder, bool unloadOnIdle)
            : this(Folder)
        {
            this.unloadOnIdle = unloadOnIdle;
        }


        protected override bool UnloadOnIdle(Activity activity)
        {
            return this.unloadOnIdle;
        }

        protected override void SaveWorkflowInstanceState(Activity rootActivity, bool unlock)
        {
            this.Save(rootActivity, unlock);
        }

        protected override Activity LoadWorkflowInstanceState(Guid instanceId)
        {
            return Load(instanceId, null);
        }

        protected override void UnlockWorkflowInstanceState(Activity rootActivity)
        {
            Guid ID = (Guid)rootActivity.GetValue(Activity.ActivityContextGuidProperty);
            string FileName = this.CreateFileName(ID);
            using (FileStream FS = new FileStream(FileName, FileMode.Open))
            {
                File.SetAttributes(FileName, FileAttributes.Normal);
            }
        }

        protected override void SaveCompletedContextActivity(Activity activity)
        {
            this.Save(activity, true);
        }

        protected override Activity LoadCompletedContextActivity(Guid scopeId, Activity outerActivity)
        {
            return Load(scopeId, outerActivity);
        }








    }

}


